﻿using UnityEngine;
using System;
using System.Collections;
using System.Collections.Generic;

public class ProgressData
{
    #region Structures
    public struct LevelData
    {
        public LevelSettings.LevelName LevelName;
        public bool Finished;
        public int BestTime;
        public int BestScore;
        public CheckpointData[] Checkpoints;
    };

    public struct CheckpointData
    {
        public int Order;
        public List<int> CollectedItems;
    };
    #endregion

    #region Data
    // Score
    private int m_RegisteredScore;
    public int RegisteredScore
    {
        get { return m_RegisteredScore; }
    }
    private int m_Score;
    public int Score
    {
        get { return m_Score; }
    }
    private int m_RegisteredCurrentLevelScore;
    public int RegisteredCurrentLevelScore
    {
        get { return m_RegisteredCurrentLevelScore; }
    }
    private int m_CurrentLevelScore;
    public int CurrentLevelScore
    {
        get { return m_CurrentLevelScore; }
    }

    // Time
    private int m_BeginTime;
    private int m_PlayTime;
    public int PlayTime
    {
        get { return m_PlayTime; }
    }
    private int m_CurrentLevelTime;
    public int CurrentLevelTime
    {
        get { return m_CurrentLevelTime; }
    }

    // Current level
    private LevelSettings.LevelName m_CurrentLevelName;
    public LevelSettings.LevelName CurrentLevelName
    {
        get { return m_CurrentLevelName; }
    }

    // Current and last checkpoints
    private int m_CurrentCheckPointOrder;
    public int CurrentCheckPointOrder
    {
        get { return m_CurrentCheckPointOrder; }
    }
    private CheckPoint m_CurrentCheckPoint;
    public CheckPoint CurrentCheckPoint
    {
        get
        {
            if (m_CurrentCheckPoint == null)
            {
                if (m_CurrentCheckPointOrder == -1)
                    return null;
                m_CurrentCheckPoint = GameObject.Find("CheckPoint" + m_CurrentCheckPointOrder).GetComponent<CheckPoint>();
                m_CurrentCheckPoint.SetCurrent(true);
            }
            return m_CurrentCheckPoint;
        }
    }

    private int m_LastCheckPointOrder;
    public int LastCheckPointOrder
    {
        get { return m_LastCheckPointOrder; }
    }
    private CheckPoint m_LastCheckPoint;
    public CheckPoint LastCheckPoint
    {
        get
        {
            if (m_LastCheckPoint == null)
            {
                if (m_LastCheckPointOrder == -1)
                    return null;
                m_LastCheckPoint = GameObject.Find("CheckPoint" + m_LastCheckPointOrder).GetComponent<CheckPoint>();
            }
            return m_LastCheckPoint;
        }
    }

    private LevelData[] m_LevelsData;
    #endregion

    #region Initialization
    public ProgressData()
    {
        InitializeCurrentLevelData();
    }

    public void Initialize()
    {
        // Initialize score
        m_Score = 0;
        m_RegisteredScore = 0;

        // Initialize play time
        m_PlayTime = 0;

        // Create new levels data
        int nbLevel = Enum.GetNames(typeof(LevelSettings.LevelName)).Length;
        m_LevelsData = new LevelData[nbLevel];

        for (int i = 0; i < nbLevel; ++i)
        {
            m_LevelsData[i].LevelName = (LevelSettings.LevelName)i;
            m_LevelsData[i].Finished = false;

            int nbCP = LevelSettings.NbCheckpoints[i];
            m_LevelsData[i].Checkpoints = new CheckpointData[nbCP];

            for (int j = 0; j < nbCP; ++j)
            {
                m_LevelsData[i].Checkpoints[j].Order = j;
                m_LevelsData[i].Checkpoints[j].CollectedItems = new List<int>();
            }
        }

        InitializeCurrentLevelData();

        // Affect initial current level value
        m_CurrentLevelName = LevelSettings.LevelName.Intro;
    }

    private void InitializeCurrentLevelData()
    {
        // Initialize score
        m_RegisteredCurrentLevelScore = 0;
        m_CurrentLevelScore = 0;
        Messenger.Broadcast<int>(Message.eScoreUpdated, m_Score);

        // Initialize time
        m_CurrentLevelTime = 0;

        // Initialize checkpoint
        m_CurrentCheckPoint = null;
        m_CurrentCheckPointOrder = -1;
        m_LastCheckPoint = null;
        m_LastCheckPointOrder = -1;
    }

    public void HackLaunch()
    {
        m_CurrentCheckPointOrder = -1;
        m_LastCheckPointOrder = -1;
    }
    #endregion

    #region Save / Load
    public void SetCurrentLevelName(LevelSettings.LevelName lvlName)
    {
        m_CurrentLevelName = lvlName;
    }

    public void SetCurrentCheckPoint(int order)
    {
        m_CurrentCheckPointOrder = order;
    }

    public void SetLastCheckPoint(int order)
    {
        m_LastCheckPointOrder = order;
    }

    public void SetLevelsData(LevelData[] data)
    {
        m_LevelsData = data;
    }

    public void InitializeScore(int score)
    {
        m_Score = score;
        m_RegisteredScore = score;
        Messenger.Broadcast<int>(Message.eScoreUpdated, m_Score);
    }

    public void InitializePlayTime(int time)
    {
        m_PlayTime = time;
    }

    public void InitializeCurrentLevelTime(int time)
    {
        m_CurrentLevelTime = time;
    }

    public void InitializeCurrentLevelScore(int score)
    {
        m_RegisteredCurrentLevelScore = score;
        m_CurrentLevelScore = score;
    }

    // get the current progress data (for saving)
    public LevelData[] GetLevelsData(out int size)
    {
        if (m_LevelsData == null)
            Initialize();

        size = m_LevelsData.Length;
        return m_LevelsData;
    }
    #endregion

    #region Score
    public void UpdateRegisteredScore(int add)
    {
        m_RegisteredScore += add;
        m_RegisteredCurrentLevelScore += add;
        UpdateScore(add);
    }

    public void UpdateScore(int add)
    {
        m_Score += add;
        m_CurrentLevelScore += add;
        Messenger.Broadcast<int>(Message.eScoreUpdated, m_Score);
    }

    // Called when the player loses
    public void BackToRegisteredScore()
    {
        UpdateScore(m_RegisteredScore - m_Score);
    }
    #endregion

    #region Checkpoint
    // New checkpoint
    public void NewCheckpoint(CheckPoint cp)
    {
        // Save the time
        m_CurrentLevelTime += (int)Time.time - m_BeginTime;
        m_PlayTime += (int)Time.time - m_BeginTime;
        m_BeginTime = (int)Time.time;

        // Is it the last one ?
        if (m_CurrentCheckPoint == null || cp.Order > m_LastCheckPoint.Order)
        {
            m_LastCheckPoint = cp;
            m_LastCheckPointOrder = cp.Order;

            // The player progress, he wins some points
            UpdateScore(m_LastCheckPoint.Points);
        }

        // Save the progress of the last check point
        if (m_CurrentCheckPoint != null)
        {
            m_CurrentCheckPoint.SetCurrent(false);
            m_LevelsData[(int)m_CurrentLevelName].Checkpoints[m_CurrentCheckPointOrder].CollectedItems = m_CurrentCheckPoint.SaveProgress();
        }

        // Assign the new check point as the current one
        m_CurrentCheckPoint = cp;
        m_CurrentCheckPointOrder = cp.Order;
        Debug.Log("New checkpoint : " + m_CurrentCheckPoint.name);
        // Save its progress too
        m_CurrentCheckPoint.SaveProgress();

        // Save the score
        m_RegisteredScore = m_Score;
        m_RegisteredCurrentLevelScore = m_CurrentLevelScore;
    }

    // Call during level initialization if the player choose continue
    public void SetProgressForAllCheckpoints(CheckPoint[] checkpoints)
    {
        if(checkpoints == null || checkpoints.Length == 0)
            return;

        LevelData ld = m_LevelsData[(int)m_CurrentLevelName];
        foreach (CheckPoint cp in checkpoints)
        {
            Debug.Log("Order = " + cp.Order);
            Debug.Log("Set to checkpoint " + cp.Order + " " + ld.Checkpoints[cp.Order].CollectedItems.Count + "collected items id");
            cp.SetItemsStatus(ld.Checkpoints[cp.Order].CollectedItems);
        }
    }
    #endregion

    #region Time
    public void GamePause(bool pause)
    {
        if (pause)
        {
            m_PlayTime += (int)Time.time - m_BeginTime;
            m_CurrentLevelTime += (int)Time.time - m_BeginTime;
        }
        else
        {
            m_BeginTime = (int)Time.time;
        }
    }
    #endregion

    #region Level
    public void NewLevel()
    {
        InitializeCurrentLevelData();

        // If the level is finished, that mean that the player want to retry the level
        // We need to remove the level best score to from the total
        LevelData ld = m_LevelsData[(int)m_CurrentLevelName];
        if (ld.Finished)
        {
            UpdateRegisteredScore(-ld.BestScore);
        }
    }

    public void LevelBegin()
    {
        m_BeginTime = (int)Time.time;
    }

    public void LevelEnd()
    {
        // register the playing time
        m_PlayTime += (int)(Time.time - m_BeginTime);
        // Register the level time
        m_CurrentLevelTime += (int)(Time.time - m_BeginTime);

        LevelData ld = m_LevelsData[(int)m_CurrentLevelName];
        ld.Finished = true;
        // Is it the best time ?
        if (ld.BestTime == 0 || ld.BestTime > m_CurrentLevelTime)
        {
            ld.BestTime = m_CurrentLevelTime;
        }

        // New best score ?
        if (ld.BestScore == 0 || ld.BestScore < m_CurrentLevelScore)
        {
            Debug.Log("New best score : " + m_CurrentLevelScore);
            ld.BestScore = m_CurrentLevelScore;
        }

        m_LevelsData[(int)m_CurrentLevelName] = ld;
    }

    // The player finished the level
    public void NextLevel()
    {
        m_CurrentCheckPointOrder = -1;
        m_CurrentCheckPoint = null;
        m_LastCheckPointOrder = -1;
        m_LastCheckPoint = null;
        m_CurrentLevelName++;
    }

    public bool GameFinished()
    {
        bool result = true;

        foreach (LevelData lvlData in m_LevelsData)
        {
            if (!lvlData.Finished)
            {
                Debug.Log(lvlData.LevelName.ToString());
                result = false;
            }
        }

        return result;
    }
    #endregion
}