﻿using UnityEngine;
using System.Collections;

public class ComputerScreen : MonoBehaviour
{
    public enum Anchor
    {
        UpperLeft,
        UpperCenter,
        UpperRight,
        Left,
        Center,
        Right,
        DownLeft,
        DownCenter,
        DownRight
    };

    public Camera m_Camera;

    // Texture
    public Texture2D m_Texture;
    public bool m_Transparent;
    // Fullscreen, FullWidth, FullHeight or Custom mode
    public bool m_FullWidth;
    public bool m_FullHeight;
    // Keep original texture ratio (not for fullscreen)
    public bool m_KeepRatio;
    // Only for Custom mode
    public bool m_KeepWidth;
    // Anchor (not for fullscreen)
    public Anchor m_Anchor;
    // Offset with the anchor comparing to the screen size (in unity unit) (not for fullscreen)
    public Vector2 m_Offset;
    // Size comparing to the screen size (in unity unit) (not for fullscreen)
    public Vector2 m_Size;
    // Plan unit size (for unit unity calculs)
    public float m_PlanUnitSize = 10f; // The generic unity plan size is 10 unity unit

    void OnEnable()
    {
        if (m_Camera == null) m_Camera = Camera.main;
        // Change the shader if necessary
        if (m_Transparent)
        {
            renderer.material.shader = Shader.Find("Transparent/Diffuse");
            Color c = renderer.material.color;
            c.a = 1f;
            renderer.material.color = c;
        }
        // Apply texture
        renderer.material.SetTexture("_MainTex", m_Texture);

        float distance = Vector3.Distance(m_Camera.transform.position, transform.position);
        float height, width;

        // Calculate the height (in unity unit)
        float heightUnit = Mathf.Tan(m_Camera.fieldOfView * 0.5f * Mathf.Deg2Rad) * distance / m_PlanUnitSize;
        heightUnit *= 2f;
        // Get the screen ratio
        float screenRatio = (float)Screen.width / Screen.height;
        // Get the width (in unity unit)
        float widthUnit = heightUnit * screenRatio;

        // Set size
        // -> FullScreen
        if (m_FullHeight && m_FullWidth)
        {
            // => Ignore anchor, size and offset
            width = widthUnit;
            height = heightUnit;
            // Set directly the correct size for the plane
            transform.localScale = new Vector3(width, 1f, height);

            // Set position (center anyway)
            Vector3 pos = (m_Camera.transform.position + m_Camera.transform.forward) * distance;
            //pos *= transform.localPosition.y;
            //transform.position = pos;
            //transform.position = new Vector3(m_Camera.transform.position.x, m_Camera.transform.position.y, transform.position.z);
        }
        // -> Full height
        else if(m_FullHeight)
        {
            // The height is the all height unit of screen
            height = heightUnit;
            // Get the width
            if (m_KeepRatio)
            {
                width = height * m_Texture.width / m_Texture.height;
            }
            else
            {
                width = m_Size.x * widthUnit;
            }

            // Set the correct size for the plane
            transform.localScale = new Vector3(width, 1f, height);

            // Set position (ignore upper or down, vertically centered anyway)
            switch (m_Anchor)
            {
            case Anchor.UpperLeft: case Anchor.Left: case Anchor.DownLeft:
                transform.position = new Vector3(m_Camera.transform.position.x - widthUnit * 0.5f * m_PlanUnitSize + transform.localScale.x * 0.5f * m_PlanUnitSize, m_Camera.transform.position.y, transform.position.z);
                break;

            case Anchor.UpperCenter: case Anchor.Center: case Anchor.DownCenter:
                transform.position = new Vector3(m_Camera.transform.position.x, m_Camera.transform.position.y, transform.position.z);
                break;

            case Anchor.UpperRight: case Anchor.Right: case Anchor.DownRight:
                transform.position = new Vector3(m_Camera.transform.position.x + widthUnit * 0.5f * m_PlanUnitSize - transform.localScale.x * 0.5f * m_PlanUnitSize, m_Camera.transform.position.y, transform.position.z);
                break;
            }
        }
        else if (m_FullWidth)
        {
            // The width is the all width unit of screen
            width = widthUnit;
            // Get the height
            if (m_KeepRatio)
            {
                height = width * m_Texture.height / m_Texture.width;
            }
            else
            {
                height = m_Size.y * heightUnit;
            }

            // Set the correct size for the plane
            transform.localScale = new Vector3(width, 1f, height);

            // Set position (ignore left or right, horizontally centered anyway)
            switch (m_Anchor)
            {
            case Anchor.UpperLeft: case Anchor.UpperCenter: case Anchor.UpperRight:
                transform.position = new Vector3(m_Camera.transform.position.x, m_Camera.transform.position.y + heightUnit * 0.5f * m_PlanUnitSize - transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                break;

            case Anchor.Left: case Anchor.Center: case Anchor.Right:
                transform.position = new Vector3(m_Camera.transform.position.x, m_Camera.transform.position.y, transform.position.z);
                break;

            case Anchor.DownLeft: case Anchor.DownCenter: case Anchor.DownRight:
                transform.position = new Vector3(m_Camera.transform.position.x, m_Camera.transform.position.y - heightUnit * 0.5f * m_PlanUnitSize + transform.localScale.z * 0.5f * m_PlanUnitSize, transform.position.z);
                break;
            }
        }
        else
        {
            if (m_KeepRatio)
            {
                if (m_KeepWidth)
                {
                    width = m_Size.x * widthUnit;
                    height = width * m_Texture.height / m_Texture.width;
                }
                else
                {
                    height = m_Size.y * heightUnit;
                    width = height * m_Texture.width / m_Texture.height;
                }
            }
            else
            {
                width = m_Size.x * widthUnit;
                height = m_Size.y * heightUnit;
            }

            // Set the correct size for the plane
            transform.localScale = new Vector3(width, 1f, height);

            // Set Position
            switch (m_Anchor)
            {
                case Anchor.UpperLeft:
                    transform.position = new Vector3(m_Camera.transform.position.x - widthUnit * 0.5f * m_PlanUnitSize + transform.localScale.x * 0.5f * m_PlanUnitSize + m_Offset.x * widthUnit, m_Camera.transform.position.y + heightUnit * 0.5f * m_PlanUnitSize - transform.localScale.z * 0.5f * m_PlanUnitSize - m_Offset.y * heightUnit, transform.position.z);
                    break;

                case Anchor.UpperCenter:
                    transform.position = new Vector3(m_Camera.transform.position.x + m_Offset.x * widthUnit, m_Camera.transform.position.y + heightUnit * 0.5f * m_PlanUnitSize - transform.localScale.z * 0.5f * m_PlanUnitSize - m_Offset.y * heightUnit, transform.position.z);
                    break;

                case Anchor.UpperRight:
                    transform.position = new Vector3(m_Camera.transform.position.x + widthUnit * 0.5f * m_PlanUnitSize - transform.localScale.x * 0.5f * m_PlanUnitSize - m_Offset.x * widthUnit, m_Camera.transform.position.y + heightUnit * 0.5f * m_PlanUnitSize - transform.localScale.z * 0.5f * m_PlanUnitSize - m_Offset.y * heightUnit, transform.position.z);
                    break;

                case Anchor.Left:
                    transform.position = new Vector3(m_Camera.transform.position.x - widthUnit * 0.5f * m_PlanUnitSize + transform.localScale.x * 0.5f * m_PlanUnitSize + m_Offset.x * widthUnit, m_Camera.transform.position.y + m_Offset.y * heightUnit, transform.position.z);
                    break;

                case Anchor.Center:
                    transform.position = new Vector3(m_Camera.transform.position.x + m_Offset.x * widthUnit, m_Camera.transform.position.y + m_Offset.y * heightUnit, transform.position.z);
                    break;

                case Anchor.Right:
                    transform.position = new Vector3(m_Camera.transform.position.x + widthUnit * 0.5f * m_PlanUnitSize - transform.localScale.x * 0.5f * m_PlanUnitSize - m_Offset.x * widthUnit, m_Camera.transform.position.y + m_Offset.y * heightUnit, transform.position.z);
                    break;

                case Anchor.DownLeft:
                    transform.position = new Vector3(m_Camera.transform.position.x - widthUnit * 0.5f * m_PlanUnitSize + transform.localScale.x * 0.5f * m_PlanUnitSize + m_Offset.x * widthUnit, m_Camera.transform.position.y - heightUnit * 0.5f * m_PlanUnitSize + transform.localScale.z * 0.5f * m_PlanUnitSize + m_Offset.y * heightUnit, transform.position.z);
                    break;

                case Anchor.DownCenter:
                    transform.position = new Vector3(m_Camera.transform.position.x + m_Offset.x * widthUnit, m_Camera.transform.position.y - heightUnit * 0.5f * m_PlanUnitSize + transform.localScale.z * 0.5f * m_PlanUnitSize + m_Offset.y * heightUnit, transform.position.z);
                    break;

                case Anchor.DownRight:
                    transform.position = new Vector3(m_Camera.transform.position.x + widthUnit * 0.5f * m_PlanUnitSize - transform.localScale.x * 0.5f * m_PlanUnitSize - m_Offset.x * widthUnit, m_Camera.transform.position.y - heightUnit * 0.5f * m_PlanUnitSize + transform.localScale.z * 0.5f * m_PlanUnitSize + m_Offset.y * heightUnit, transform.position.z);
                    break;
            }
        }
    }
}