using System;
using Global;
using TMPro;
using UnityEngine;
using UnityEngine.Serialization;
using UnityEngine.UI;

namespace Menu
{
    public enum MenuState
    {
        Settings,
        Shop,
        Customize,
        LevelSelect
    }

    public class MenuManager : MonoBehaviour
    {
        public static MenuManager instance;
        [FormerlySerializedAs("_levelMenuPanel")] [SerializeField] private GameObject levelMenuPanel;
        private GameObject[] _levelButtons;
        [FormerlySerializedAs("_scrollRect")] [SerializeField] private ScrollRect scrollRect;
        [FormerlySerializedAs("_SettingsPanel")] [SerializeField] private GameObject settingsPanel;
        [FormerlySerializedAs("_ShopPanel")] [SerializeField] private GameObject shopPanel;
        [FormerlySerializedAs("_CustomizePanel")] [SerializeField] private GameObject customizePanel;
        [FormerlySerializedAs("_starGrey")] [SerializeField] private Sprite starGrey;
        [FormerlySerializedAs("_starYellow")] [SerializeField] private Sprite starYellow;

        private void Awake()
        {
            if (instance == null)
            {
                instance = this;
            }
            else
            {
                Destroy(gameObject);
            }
        }

        public void Start()
        {   // init objects
            customizePanel.GetComponent<Canvas>().renderMode = RenderMode.ScreenSpaceOverlay;
            shopPanel.GetComponent<Canvas>().renderMode = RenderMode.ScreenSpaceOverlay;
            settingsPanel.GetComponent<Canvas>().renderMode = RenderMode.ScreenSpaceOverlay;
            levelMenuPanel.GetComponentInChildren<Button>().onClick.AddListener(() => StartLevel());
            this.ChangeMenuState(MenuState.LevelSelect);
            _levelButtons = GameObject.FindGameObjectsWithTag("LevelButton");
            Array.Sort(_levelButtons, (a, b) =>
                int.Parse(a.transform.GetChild(0).GetComponent<TextMeshProUGUI>().text)
                    .CompareTo(int.Parse(b.transform.GetChild(0).GetComponent<TextMeshProUGUI>().text)));
            for (int i = 0; i < _levelButtons.Length; i++)
            {
                int levelIndex = i; // Capture the current value of i
                _levelButtons[i].GetComponent<Button>().onClick.AddListener(() => DisplayLevelPanel(levelIndex));
            }
            
            // disable buttons if the previous level does not have at least one star collected
            //except for the first level
            for (int i = 1; i < _levelButtons.Length; i++)
            {
                bool[] starsCollected = PlayerStatsManager.instance.GetLevelStarsCollected(i - 1);
                bool hasAtLeastOneStar = false;
                foreach (bool star in starsCollected)
                {
                    if (star)
                    {
                        hasAtLeastOneStar = true;
                        break;
                    }
                }
                if (!hasAtLeastOneStar)
                {
                    _levelButtons[i].GetComponent<Button>().interactable = false;
                }
            }
        }

        public void DisplayLevelPanel(int levelIndex)
        {
            levelMenuPanel.SetActive(true);
            levelMenuPanel.transform.GetChild(0).GetComponent<TextMeshProUGUI>().text = "Level " + (levelIndex + 1);
            // get the data from the PlayerStatsManager about the stars collected in this level
            bool[] starsCollected = PlayerStatsManager.instance.GetLevelStarsCollected(levelIndex);
            // update the stars display in the level menu panel by using child(1) and child(2) and child(3)
            for (int i = 0; i < 3; i++)
            {
                if (starsCollected[i])
                {
                    levelMenuPanel.transform.GetChild(i + 1).GetComponent<Image>().sprite = starYellow;
                }
                else
                {
                    levelMenuPanel.transform.GetChild(i + 1).GetComponent<Image>().sprite = starGrey;
                }
            }
            // Depending on the number of stars collected, update the level menu panel score (Child 5)
            int starsCount = 0;
            foreach (bool star in starsCollected)
            {
                if (star)
                    starsCount++;
            }
            switch (starsCount)
            {
                case 0:
                    levelMenuPanel.transform.GetChild(5).GetComponent<TextMeshProUGUI>().text = "Score to reach : " + PlayerStatsManager.instance.GetLevelScoreToReach(levelIndex, starsCount);
                    break;
                case 1:
                    levelMenuPanel.transform.GetChild(5).GetComponent<TextMeshProUGUI>().text = "Score to reach : " + PlayerStatsManager.instance.GetLevelScoreToReach(levelIndex, starsCount);
                    break;
                case 2:
                    levelMenuPanel.transform.GetChild(5).GetComponent<TextMeshProUGUI>().text = "Score to reach : " + PlayerStatsManager.instance.GetLevelScoreToReach(levelIndex, starsCount);
                    break;
                default:
                    levelMenuPanel.transform.GetChild(5).GetComponent<TextMeshProUGUI>().text = "Best score : " + PlayerStatsManager.instance.GetLevelBestScore(levelIndex);
                    break;
            }
            // Set the panel child of the clicked button
            levelMenuPanel.transform.SetParent(_levelButtons[levelIndex].transform, false);
            // set position to 0 360 y au dessus
            levelMenuPanel.transform.localPosition = new Vector3(0, 360, 0);
            // change scroll rect value to display the panel in the center of the screen
            // 
            scrollRect.verticalNormalizedPosition = (float)levelIndex / (_levelButtons.Length - 1);
        }
    
        public void CloseLevelPanel()
        {
            levelMenuPanel.SetActive(false);
        }
    
        public void StartLevel()
        {
            int levelIndex = Array.IndexOf(_levelButtons, levelMenuPanel.transform.parent.gameObject);
            // Load the level scene here
            // Example: SceneManager.LoadScene("Level" + levelIndex);
            PlayerStatsManager.instance.currentLevelToPlay = PlayerStatsManager.instance.GetLevelByIndex(levelIndex);
            //UnityEngine.SceneManagement.SceneManager.LoadScene("Game");
            SceneTransitionManager.instance.GoToScene("Game");
        }
    
        public void ChangeMenuState(MenuState newState)
        {
            levelMenuPanel.SetActive(false);
            settingsPanel.SetActive(newState == MenuState.Settings);
            shopPanel.SetActive(newState == MenuState.Shop);
            customizePanel.SetActive(newState == MenuState.Customize);
        }

        public void ShowCustomizePanel()
        {
            this.ChangeMenuState(MenuState.Customize);
        }

        public void ShowShopPanel()
        {
            this.ChangeMenuState(MenuState.Shop);
        }
    
        public void ShowSettingsPanel()
        {
            this.ChangeMenuState(MenuState.Settings);
        }
    
        public void ReturnToLevelSelect()
        {
            this.ChangeMenuState(MenuState.LevelSelect);
        }
    }
}