#if UNITY_EDITOR
using System;
using UnityEngine;
using UnityEditor;
using System.IO;
using Scriptables;

public class LevelImporter : MonoBehaviour
{
    [MenuItem("Tools/Import Levels From JSON")]
    public static void ImportLevels()
    {
        string path = EditorUtility.OpenFilePanel("Select Level JSON", "", "json");
        if (string.IsNullOrEmpty(path)) return;
        // Delete all existing Level assets
        string levelsFolder = "Assets/Resources/Levels/";
        if (Directory.Exists(levelsFolder))
        {
            string[] existingAssets = Directory.GetFiles(levelsFolder, "Level_*.asset");
            foreach (var assetPath in existingAssets)
            {
                AssetDatabase.DeleteAsset(assetPath);
            }
        }

        string json = File.ReadAllText(path);
        LevelData[] data = JsonHelper.FromJson<LevelData>(json);

        LevelSet levelSet = ScriptableObject.CreateInstance<LevelSet>();

        foreach (var lvl in data)
        {
            LevelComposition comp = ScriptableObject.CreateInstance<LevelComposition>();
            comp.levelIndex = lvl.level;
            comp.TimeBeforeEnd = lvl.duration;
            // Load enemy prefabs
            comp.EnemiesPrefabs = new GameObject[lvl.monsters.Length];
            for (int i = 0; i < lvl.monsters.Length; i++)
            {
                string enemyPath = $"Assets/Quirky Series Ultimate/FREE/Prefabs/{lvl.monsters[i]}.prefab";
                GameObject enemyPrefab = AssetDatabase.LoadAssetAtPath<GameObject>(enemyPath);
                if (enemyPrefab != null)
                {
                    comp.EnemiesPrefabs[i] = enemyPrefab;
                }
                else
                {
                    Debug.LogWarning($"Enemy prefab not found at path: {enemyPath}");
                }
            }
            comp.scoreForStar1 = lvl.stars[0];
            comp.scoreForStar2 = lvl.stars[1];
            comp.scoreForStar3 = lvl.stars[2];
            
            comp.displayTuto = lvl.displayTuto;
            levelSet.levels.Add(comp);

            AssetDatabase.CreateAsset(comp, $"Assets/Resources/Levels/Level_{lvl.level}.asset");
        }
        AssetDatabase.CreateAsset(levelSet, "Assets/Resources/Levels/LevelSet.asset");
        AssetDatabase.SaveAssets();

        Debug.Log($"Imported {data.Length} levels successfully.");
    }
}

public static class JsonHelper
{
    public static T[] FromJson<T>(string json)
    {
        string wrapped = "{\"Items\":" + json + "}";
        Wrapper<T> wrapper = JsonUtility.FromJson<Wrapper<T>>(wrapped);
        return wrapper.Items;
    }

    [Serializable]
    private class Wrapper<T>
    {
        public T[] Items;
    }
}
#endif
