#if UNITY_EDITOR
using System.IO;
using System.Collections.Generic;
using UnityEngine;
using UnityEditor;

public class AssetPreviewExporter : EditorWindow
{
    [SerializeField]
    private List<GameObject> prefabs = new List<GameObject>();

    [SerializeField]
    private List<Material> materials = new List<Material>();

    [SerializeField]
    private string outputFolder = "Assets/AssetPreviews";

    private SerializedObject so;
    private SerializedProperty prefabsProperty;
    private SerializedProperty materialsProperty;

    [MenuItem("Tools/Exporter les previews d'assets")]
    private static void ShowWindow()
    {
        var window = GetWindow<AssetPreviewExporter>("Export previews");
        window.minSize = new Vector2(450, 300);
    }

    private void OnEnable()
    {
        so = new SerializedObject(this);
        prefabsProperty = so.FindProperty("prefabs");
        materialsProperty = so.FindProperty("materials");
    }

    private void OnGUI()
    {
        so.Update();

        EditorGUILayout.LabelField("Prefabs à exporter", EditorStyles.boldLabel);
        EditorGUILayout.PropertyField(prefabsProperty, includeChildren: true);

        EditorGUILayout.Space(10);

        EditorGUILayout.LabelField("Materials à exporter", EditorStyles.boldLabel);
        EditorGUILayout.PropertyField(materialsProperty, includeChildren: true);

        EditorGUILayout.Space(15);

        EditorGUILayout.LabelField("Dossier de sortie (relatif au projet)", EditorStyles.boldLabel);
        outputFolder = EditorGUILayout.TextField("Output folder", outputFolder);

        if (GUILayout.Button("Choisir un dossier..."))
        {
            var absPath = EditorUtility.OpenFolderPanel("Choisir le dossier de sortie", Application.dataPath, "");
            if (!string.IsNullOrEmpty(absPath))
            {
                // Convertit un chemin absolu en chemin relatif au projet (Assets/...)
                var projectPath = Application.dataPath.Substring(0, Application.dataPath.Length - "Assets".Length);
                if (absPath.StartsWith(projectPath))
                {
                    outputFolder = absPath.Substring(projectPath.Length).Replace("\\", "/");
                }
                else
                {
                    Debug.LogWarning("Le dossier choisi doit être dans le projet Unity.");
                }
            }
        }

        EditorGUILayout.Space(20);

        if (GUILayout.Button("Exporter les previews en PNG"))
        {
            ExportPreviews();
        }

        so.ApplyModifiedProperties();
    }

    private void ExportPreviews()
    {
        bool hasPrefabs = prefabs != null && prefabs.Count > 0;
        bool hasMaterials = materials != null && materials.Count > 0;

        if (!hasPrefabs && !hasMaterials)
        {
            Debug.LogWarning("Aucun prefab ou material à exporter.");
            return;
        }

        if (string.IsNullOrEmpty(outputFolder))
        {
            Debug.LogWarning("Dossier de sortie invalide.");
            return;
        }

        // Crée le dossier si besoin (relatif au projet)
        if (!Directory.Exists(outputFolder))
        {
            Directory.CreateDirectory(outputFolder);
        }

        if (hasPrefabs)
        {
            ExportAssetList(prefabs, "_prefab");
        }

        if (hasMaterials)
        {
            ExportAssetList(materials, "_mat");
        }

        AssetDatabase.Refresh();
        Debug.Log("Export des previews terminé.");
    }

    private void ExportAssetList<T>(List<T> assets, string nameSuffix) where T : UnityEngine.Object
    {
        foreach (var asset in assets)
        {
            if (asset == null)
                continue;

            // Essaie d'abord d'obtenir la preview "pleine"
            Texture2D tex = AssetPreview.GetAssetPreview(asset);

            // Si pas encore générée, fallback sur la mini vignette
            if (tex == null)
            {
                tex = AssetPreview.GetMiniThumbnail(asset) as Texture2D;
            }

            if (tex == null)
            {
                Debug.LogWarning($"Impossible de générer une preview pour : {asset.name}");
                continue;
            }

            // Rend la texture lisible via un RenderTexture
            Texture2D readable = MakeReadable(tex);
            if (readable == null)
            {
                Debug.LogWarning($"Échec de création d'une texture lisible pour : {asset.name}");
                continue;
            }

            byte[] pngData = readable.EncodeToPNG();
            UnityEngine.Object.DestroyImmediate(readable);

            if (pngData == null || pngData.Length == 0)
            {
                Debug.LogWarning($"Erreur à l'encodage PNG pour : {asset.name}");
                continue;
            }

            string safeName = MakeFileNameSafe(asset.name);
            if (!string.IsNullOrEmpty(nameSuffix))
                safeName += nameSuffix;

            string filePath = Path.Combine(outputFolder, safeName + ".png");
            filePath = filePath.Replace("\\", "/");

            File.WriteAllBytes(filePath, pngData);

            Debug.Log($"Preview exportée : {filePath}");
        }
    }

    private static string MakeFileNameSafe(string name)
    {
        foreach (char c in Path.GetInvalidFileNameChars())
        {
            name = name.Replace(c, '_');
        }
        return name;
    }

    /// <summary>
    /// Convertit n'importe quelle Texture en Texture2D lisible (RGBA32) via un RenderTexture.
    /// </summary>
    private static Texture2D MakeReadable(Texture tex)
    {
        if (tex == null)
            return null;

        var rt = RenderTexture.GetTemporary(tex.width, tex.height, 0, RenderTextureFormat.ARGB32);

        // Copie la texture dans le RenderTexture
        Graphics.Blit(tex, rt);

        // Lit les pixels dans une nouvelle Texture2D lisible
        var previous = RenderTexture.active;
        RenderTexture.active = rt;

        var readable = new Texture2D(tex.width, tex.height, TextureFormat.RGBA32, false);
        readable.ReadPixels(new Rect(0, 0, tex.width, tex.height), 0, 0);
        readable.Apply();

        RenderTexture.active = previous;
        RenderTexture.ReleaseTemporary(rt);

        return readable;
    }
}
#endif