using System;
using Global;
using UnityEngine;
using UnityEngine.InputSystem;
using UnityEngine.Serialization;
using UnityEngine.UI;
using Random = UnityEngine.Random;

namespace Game
{
    public enum State
    {
        
        Playing,
        Tutorial,
        Pause,
        GameOver
    }

    public class GameManager : MonoBehaviour
    {
        public static GameManager instance;
        [FormerlySerializedAs("TimeBeforeEnd")] [SerializeField] public float timeBeforeEnd = 90f;
        private float _initialTimeBeforeEnd;
        [FormerlySerializedAs("EnemiesPrefabs")] [SerializeField] private GameObject[] enemiesPrefabs;
        [FormerlySerializedAs("WaypointsLeft")] [SerializeField] private GameObject waypointsLeft;
        [FormerlySerializedAs("WaypointsRight")] [SerializeField] private GameObject waypointsRight;
        [SerializeField] private GameObject TimeSlider;
        public static event Action<GameObject[]> OnGameStart;
        public static event Action OnGameEnd;
        private bool _gameOver;
    
        // State machine system for UI and game status
        private State _state;
        private GameObject _canvasPause;
        private GameObject _canvasPlay;
        private GameObject _canvasGameOver;
        private GameObject _canvasTutorial;
        private PlayerInput _playerInput;
        //TODO modify score management to fit with enemies
        [FormerlySerializedAs("Score")] public int score;
    
    
        public void Awake()
        {
            if (instance == null)
            {
                instance = this;
            }
            else
            {
                Destroy(gameObject);
            }
        }
        public void Start()
        {
            _playerInput = GameObject.Find("GameCursor").GetComponent<PlayerInput>();
            _canvasPause = GameObject.FindGameObjectWithTag("CanvasPause");
            _canvasPlay = GameObject.FindGameObjectWithTag("CanvasPlay");
            _canvasGameOver = GameObject.FindGameObjectWithTag("CanvasGameOver");
            _canvasTutorial = GameObject.FindGameObjectWithTag("CanvasTutorial");
            _canvasPause.GetComponent<Canvas>().renderMode = RenderMode.ScreenSpaceOverlay;
            _canvasPlay.GetComponent<Canvas>().renderMode = RenderMode.ScreenSpaceOverlay;
            _canvasGameOver.GetComponent<Canvas>().renderMode = RenderMode.ScreenSpaceOverlay;
            _canvasTutorial.GetComponent<Canvas>().renderMode = RenderMode.ScreenSpaceOverlay;
            //Load data from PlayerStatsManager
            if (PlayerStatsManager.instance == null)
            {
                Debug.LogWarning("PlayerStatsManager instance is null!");
                _initialTimeBeforeEnd = timeBeforeEnd;
            }
            else if (PlayerStatsManager.instance.currentLevelToPlay == null)
            {
                Debug.LogWarning("PlayerStatsManager.instance.currentLevelToPlay is null!");
                _initialTimeBeforeEnd = timeBeforeEnd;
            }
            else
            {
                timeBeforeEnd = PlayerStatsManager.instance.currentLevelToPlay.TimeBeforeEnd;
                enemiesPrefabs = PlayerStatsManager.instance.currentLevelToPlay.EnemiesPrefabs;
                _initialTimeBeforeEnd = timeBeforeEnd;
                if (PlayerStatsManager.instance.currentLevelToPlay.displayTuto != 0)
                {
                    this.ChangeState(State.Tutorial);
                    _canvasTutorial.transform.GetChild(0).GetChild(_canvasTutorial.transform.GetChild(0).childCount - 1).GetComponent<TutorialCanvas>().SetTutorial(PlayerStatsManager.instance.currentLevelToPlay.displayTuto);
                }
            }
            StartGame();
        }

        public void Update()
        { 
            if (_state == State.Pause || _state == State.Tutorial)
            { 
                // Do nothing, game is paused or in tutorial
            }
            else if (_state == State.Playing && timeBeforeEnd > 0)
            {
                timeBeforeEnd -= Time.deltaTime;
                TimeSlider.GetComponent<Slider>().value = 1 - (timeBeforeEnd / _initialTimeBeforeEnd);
            }
            else if (!_gameOver)
            {
                EndGame();
            }
        }
    
        private void StartGame()
        {
            if (_state == State.Tutorial)
            {
                return;
            }
            this.ChangeState(State.Playing);
            OnGameStart?.Invoke(enemiesPrefabs);
        }
        
        public void StartFromTutorial()
        {
            this.ChangeState(State.Playing);
            OnGameStart?.Invoke(enemiesPrefabs);
        }
    
        private void EndGame()
        {
            _gameOver = true;
            this.ChangeState(State.GameOver);
            OnGameEnd?.Invoke();
            Debug.Log("Game Over!");
        }
    
        public Transform GetWaypoints(Side side)
        {
            return side == Side.Left ? waypointsLeft.transform.GetChild(Random.Range(0, waypointsLeft.transform.childCount))
                .transform : waypointsRight.transform.GetChild(Random.Range(0, waypointsRight.transform.childCount)).transform;
        }

        private void ChangeState(State newState)
        {
            _state = newState;
            if (_state != State.Playing)
            {
                Time.timeScale = 0;
            }
            else
            {
                Time.timeScale = 1;
            }

            _canvasPause.SetActive(_state == State.Pause);
            _canvasPlay.SetActive(_state == State.Playing);
            _canvasGameOver.SetActive(_state == State.GameOver);
            _canvasTutorial.SetActive(_state == State.Tutorial);
        }

        public void ReturnToMenu()
        {
            Debug.Log("Return to Main Menu");
            //UnityEngine.SceneManagement.SceneManager.LoadScene("Menu");
            SceneTransitionManager.instance.GoToScene("Menu");
        }

        public void RestartGame()
        {
            Debug.Log("Restart Game");
            //UnityEngine.SceneManagement.SceneManager.LoadScene("Game");
            SceneTransitionManager.instance.GoToScene("Game");
        }
    
        public void TogglePause()
        {
            switch (_state)
            {
                case State.Playing:
                    ChangeState(State.Pause);
                    break;
                case State.Pause:
                    ChangeState(State.Playing);
                    break;
                case State.GameOver:
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }
    }
}