using UnityEngine;
using UnityEngine.Serialization;
using Random = UnityEngine.Random;

namespace Game
{
    public class EnemySpawner : MonoBehaviour
    {
        private bool _canSpawn = false;
        private GameObject[] _enemiesPrefabs;
        [FormerlySerializedAs("_spawnInterval")] [SerializeField] private float spawnInterval = 2f;
        [FormerlySerializedAs("_spawnSide")] [SerializeField] private Side spawnSide = Side.Left;

        public void Awake()
        {
            GameManager.OnGameStart += AllowSpawning;
            GameManager.OnGameEnd += DisallowSpawning;
        }

        private void AllowSpawning(GameObject[] enemiesPrefabs)
        {
            _canSpawn = true;
            _enemiesPrefabs = enemiesPrefabs;
        }

        private void DisallowSpawning()
        {
            _canSpawn = false;
        }

        public void Update()
        {
            if (!_canSpawn) return;
            spawnInterval -= Time.deltaTime;
            if (!(spawnInterval <= 0f)) return;
            SpawnEnemy();
            spawnInterval = Random.Range(1f, 3f); // Reset spawn interval
        }
    
        private void SpawnEnemy()
        {
            if (_enemiesPrefabs.Length == 0) return;
        
            var randomIndex = Random.Range(0, _enemiesPrefabs.Length);
            var startPosition = new Vector3(transform.position.x, transform.position.y, Random.Range(transform.position.z - transform.localScale.z / 2, transform.position.z + transform.localScale.z / 2));
            var go = Instantiate(_enemiesPrefabs[randomIndex], startPosition, Quaternion.identity);
            go.GetComponent<Enemy>().targetWaypoint = GameManager.instance.GetWaypoints(spawnSide == Side.Left ? Side.Right : Side.Left);
        }
    }
}
