using UnityEngine;
using UnityEngine.AI;

public class S_Mouse : MonoBehaviour
{
    private NavMeshAgent agent;
    private Transform cat;

    [Header("Fuite")]
    [SerializeField] private float fleeDistance = 10f;      // distance que la souris essaie de garder
    [SerializeField] private float triggerDistance = 8f;    // distance à partir de laquelle elle fuit
    [SerializeField] private float safeDistance = 12f;      // distance où elle se sent en sécurité
    [SerializeField] private float randomOffset = 1.5f;     // petit bruit aléatoire pour varier la direction

    [Header("Vitesse adaptative")]
    [SerializeField] private float minSpeed = 2f;
    [SerializeField] private float maxSpeed = 6f;

    private Vector3 lastFleeTarget;
    private float distanceToCat;
    private float nextCheckTime;

    void Start()
    {
        agent = GetComponent<NavMeshAgent>();

        S_CatGrabBehavior catScript = FindFirstObjectByType<S_CatGrabBehavior>();
        if (catScript != null)
            cat = catScript.transform;
        else
            Debug.LogWarning("Chat non trouvé");
    }

    void Update()
    {
        if (cat == null) return;

        distanceToCat = Vector3.Distance(transform.position, cat.position);

        // Ajuste la vitesse selon la proximité du chat
        float panicFactor = Mathf.InverseLerp(safeDistance, 0f, distanceToCat);
        agent.speed = Mathf.Lerp(minSpeed, maxSpeed, panicFactor);

        bool shouldFlee = distanceToCat < triggerDistance;
        bool hasReachedTarget = !agent.pathPending && agent.remainingDistance < 0.5f;

        // Recalcule une nouvelle destination seulement si :
        // 1. le chat est proche
        // 2. le chat s’est déplacé significativement
        // 3. la souris a atteint sa position actuelle
        if (shouldFlee && (hasReachedTarget || Time.time > nextCheckTime))
        {
            Vector3 newTarget = CalculateFleeDestination();

            if (Vector3.Distance(newTarget, lastFleeTarget) > 1.0f)
            {
                agent.SetDestination(newTarget);
                lastFleeTarget = newTarget;
            }

            nextCheckTime = Time.time + 1.0f;
        }
        else if (!shouldFlee && hasReachedTarget)
        {
            agent.ResetPath();
        }
    }

    private Vector3 CalculateFleeDestination()
    {
        // Direction opposée au chat
        Vector3 fleeDir = (transform.position - cat.position).normalized;

        fleeDir += new Vector3(Random.Range(-randomOffset, randomOffset), 0, Random.Range(-randomOffset, randomOffset));
        fleeDir.Normalize();

        Vector3 fleeTarget = transform.position + fleeDir * fleeDistance;

        NavMeshHit hit;
        if (NavMesh.SamplePosition(fleeTarget, out hit, 3f, NavMesh.AllAreas))
            return hit.position;

        return GetRandomNavMeshPosition(transform.position, fleeDistance);
    }

    private Vector3 GetRandomNavMeshPosition(Vector3 origin, float radius)
    {
        for (int i = 0; i < 5; i++)
        {
            Vector3 randomDirection = Random.insideUnitSphere * radius;
            randomDirection += origin;

            if (NavMesh.SamplePosition(randomDirection, out NavMeshHit hit, radius, NavMesh.AllAreas))
                return hit.position;
        }
        return origin;
    }

    void OnDrawGizmosSelected()
    {
        Gizmos.color = Color.red;
        Gizmos.DrawWireSphere(transform.position, triggerDistance);
        Gizmos.color = Color.green;
        Gizmos.DrawWireSphere(transform.position, safeDistance);
    }
}
