﻿using UnityEngine;
using System.Collections;
using System.Collections.Generic;

public enum GameState
{
    MainMenu,
    OptionsScreen,
    PauseScreen,
    WinScreen,
    LooseScreen,
    EndScreen,
    Running
}

public class GameManager : MonoBehaviour
{
    #region Static
    public static GameManager Instance;
    public static GameState LastGameState;
    public static GameState CurrentState { get; private set; }


    public static void BackToLastState()
    {
        if (Instance) Instance.SetState(LastGameState);
    }
    #endregion

    public My_CharacterController_v02 Character;
    public List<GameObject> Vessels = new List<GameObject>();


    public GameObject GuiMainMenu;
    public GameObject GuiOptionsScreen;
    public GameObject GuiPauseScreen;
    public GameObject GuiWinScreen;
    public GameObject GuiLooseScreen;
    public GameObject GuiEndScreen;
    public GameObject GuiInGame;

    public float lastingTime;
    public float maxTime; 

    [SerializeField]
    private int numberOfLvl;
    public  int currentLvl;

    private int reparationDone = 0;
    private int numberOfReparation;

    private Quaternion baseRotation;
    private Vector3 basePosition;

	private Vector3 currentVelocity;
	private Vector3 currentAngularVelocity;
	private Rigidbody rbCharacter;

	#region Public methods
	public void SetState(GameState state)
    {
        turnOffAllGui();

        LastGameState = CurrentState;
        CurrentState = state;


        switch (CurrentState)
        {
            case GameState.MainMenu:
                GuiMainMenu.SetActive(true);
                currentLvl = 0;
                loadLevel();
                break;
            case GameState.OptionsScreen:
                GuiOptionsScreen.SetActive(true);
                break;
            case GameState.PauseScreen:
                GuiPauseScreen.SetActive(true);

				rbCharacter = Character.transform.GetChild(0).GetComponent<Rigidbody>();
				currentVelocity = rbCharacter.velocity;
				currentAngularVelocity = rbCharacter.angularVelocity;
				rbCharacter.isKinematic = true;

				break;
            case GameState.WinScreen:
                GuiWinScreen.SetActive(true);
                break;
            case GameState.LooseScreen:
                GuiLooseScreen.SetActive(true);
                break;
            case GameState.EndScreen:
                GuiEndScreen.SetActive(true);
                break;
            case GameState.Running:
                GuiInGame.SetActive(true);

				rbCharacter = Character.transform.GetChild(0).GetComponent<Rigidbody>();
				rbCharacter.isKinematic = false;
				rbCharacter.velocity = currentVelocity;
				rbCharacter.angularVelocity = currentAngularVelocity;
				break;

            default:
                GuiMainMenu.SetActive(true);
                CurrentState = GameState.MainMenu;
                break;
        }
    }

    public void IncrementReparationState()
    {
        reparationDone++;
		print(reparationDone);

        if (reparationDone == numberOfReparation)
        {

			print("newLevel");

            if (currentLvl == numberOfLvl)
                SetState(GameState.EndScreen);
            else
            {
                SetState(GameState.WinScreen);
            }
        }
    }

    public void ResetCurrentLevel()
    {
        SetState(GameState.Running);
        loadLevel();
    }

    public void NextLevel()
    {
        currentLvl++;
        SetState(GameState.Running);
        loadLevel();
    }

	public float GetPlayerLifePoint() => Character.CurrentLife;
	public float GetPlayerMaxLifePoint() => Character.MaxLife;

	public float GetJetpackFuel() => Character.JetPackFuel;
    public float GetJetpackMaxFuel() => Character.MaxJetPackFuel;

    public float GetLastingTime() => lastingTime;
    public float GetMaxTime() => maxTime;


    #endregion

    #region Private methods
    private void turnOffAllGui()
    {
        GuiMainMenu.SetActive(false);
        GuiOptionsScreen.SetActive(false);
        GuiPauseScreen.SetActive(false);
        GuiWinScreen.SetActive(false);
        GuiLooseScreen.SetActive(false);
        GuiEndScreen.SetActive(false);
        GuiInGame.SetActive(false);
    }

    private void loadLevel()
    {
        for (int i = 0; i < Vessels.Count; i++)
        {
            if (i == currentLvl)
                Vessels[i].gameObject.SetActive(true);
            else
                Vessels[i].gameObject.SetActive(false);
        }

        Character.GetComponentInChildren<Transform>().transform.position = basePosition;
        Character.GetComponentInChildren<Transform>().transform.rotation = baseRotation;
        reparationDone = 0;
        numberOfReparation = Vessels[currentLvl].GetComponent<ReparationPointGenerator>().TotalPoint;
        lastingTime = maxTime;

		rbCharacter = Character.transform.GetChild(0).GetComponent<Rigidbody>();
		rbCharacter.isKinematic = false;
		rbCharacter.velocity = Vector3.zero;
		rbCharacter.angularVelocity = Vector3.zero;
	}
    #endregion

    #region Unity methods
    private void Awake()
    {
        if (Instance == null) Instance = this;

        turnOffAllGui();
    }

    // Use this for initialization
    private void Start()
    {
        numberOfLvl = Vessels.Count;
        CurrentState = GameState.MainMenu;
        SetState(CurrentState);
        basePosition = Character.GetComponentInChildren<Transform>().position;
        baseRotation = Character.GetComponentInChildren<Transform>().rotation;
        lastingTime = maxTime;
    }

    // Update is called once per frame
    private void Update()
    {

        if (Input.GetKeyDown(KeyCode.Escape))
            SetState(GameState.PauseScreen);

		if (GameManager.CurrentState == GameState.Running) {
			lastingTime -= Time.deltaTime;

			if (lastingTime <= 0) {
				GameManager.Instance.SetState(GameState.LooseScreen);
			}

		}
    }
    #endregion

}
